-- AI Visibility Tracker Database Schema
-- Production-ready SaaS application

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    role ENUM('admin', 'member') DEFAULT 'member',
    is_active TINYINT(1) DEFAULT 1,
    email_verified_at TIMESTAMP NULL DEFAULT NULL,
    verification_token VARCHAR(255) NULL,
    last_login_at TIMESTAMP NULL DEFAULT NULL,
    last_login_ip VARCHAR(45) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Password resets table
CREATE TABLE IF NOT EXISTS password_resets (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    token VARCHAR(255) NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Login attempts table (brute force protection)
CREATE TABLE IF NOT EXISTS login_attempts (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL,
    attempt_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    ip_address VARCHAR(45) NULL,
    INDEX idx_email (email),
    INDEX idx_time (attempt_time),
    INDEX idx_ip (ip_address)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Websites table
CREATE TABLE IF NOT EXISTS websites (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    name VARCHAR(255) NOT NULL,
    url VARCHAR(500) NOT NULL,
    description TEXT NULL,
    industry VARCHAR(100) NULL,
    target_keywords TEXT NULL,
    is_active TINYINT(1) DEFAULT 1,
    last_scan_at TIMESTAMP NULL DEFAULT NULL,
    visibility_score INT DEFAULT 0,
    recommendation_score INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_active (is_active),
    INDEX idx_last_scan (last_scan_at),
    INDEX idx_visibility (visibility_score),
    FULLTEXT idx_search (name, description, target_keywords)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AI Models table
CREATE TABLE IF NOT EXISTS ai_models (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL UNIQUE,
    display_name VARCHAR(255) NOT NULL,
    api_endpoint VARCHAR(500) NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    max_requests_per_minute INT DEFAULT 60,
    max_requests_per_hour INT DEFAULT 1000,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default AI models
INSERT IGNORE INTO ai_models (name, display_name, api_endpoint, max_requests_per_minute, max_requests_per_hour) VALUES
('openai-gpt-4', 'OpenAI GPT-4', 'https://api.openai.com/v1/chat/completions', 60, 1000),
('openai-gpt-3.5', 'OpenAI GPT-3.5', 'https://api.openai.com/v1/chat/completions', 100, 2000),
('claude-3', 'Claude 3', 'https://api.anthropic.com/v1/messages', 50, 800),
('gemini-pro', 'Gemini Pro', 'https://generativelanguage.googleapis.com/v1beta/models/gemini-pro:generateContent', 80, 1200);

-- Job Queue table
CREATE TABLE IF NOT EXISTS job_queue (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    type ENUM('recognition_test', 'recommendation_test', 'question_association') NOT NULL,
    payload JSON NOT NULL,
    website_id INT UNSIGNED NOT NULL,
    user_id INT UNSIGNED NOT NULL,
    status ENUM('pending', 'processing', 'completed', 'failed') DEFAULT 'pending',
    priority INT DEFAULT 0,
    attempts INT DEFAULT 0,
    max_attempts INT DEFAULT 3,
    result JSON NULL,
    error_message TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    scheduled_at TIMESTAMP NULL DEFAULT NULL,
    started_at TIMESTAMP NULL DEFAULT NULL,
    completed_at TIMESTAMP NULL DEFAULT NULL,
    failed_at TIMESTAMP NULL DEFAULT NULL,
    FOREIGN KEY (website_id) REFERENCES websites(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_status (status),
    INDEX idx_website_id (website_id),
    INDEX idx_user_id (user_id),
    INDEX idx_created_at (created_at),
    INDEX idx_priority (priority),
    INDEX idx_scheduled (scheduled_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AI Requests table (logging)
CREATE TABLE IF NOT EXISTS ai_requests (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    website_id INT UNSIGNED NOT NULL,
    user_id INT UNSIGNED NOT NULL,
    model_id INT UNSIGNED NOT NULL,
    request_type ENUM('recognition_test', 'recommendation_test', 'question_association') NOT NULL,
    prompt_version VARCHAR(50) NOT NULL,
    prompt_text TEXT NOT NULL,
    response_text TEXT NULL,
    response_hash VARCHAR(64) NULL,
    tokens_used INT NULL,
    processing_time_ms INT NULL,
    status ENUM('pending', 'processing', 'completed', 'failed') DEFAULT 'pending',
    error_message TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL DEFAULT NULL,
    FOREIGN KEY (website_id) REFERENCES websites(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (model_id) REFERENCES ai_models(id) ON DELETE CASCADE,
    INDEX idx_website_id (website_id),
    INDEX idx_user_id (user_id),
    INDEX idx_model_id (model_id),
    INDEX idx_request_type (request_type),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at),
    INDEX idx_response_hash (response_hash)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Analytics table
CREATE TABLE IF NOT EXISTS analytics (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    website_id INT UNSIGNED NOT NULL,
    model_id INT UNSIGNED NOT NULL,
    visibility_score INT DEFAULT 0,
    mention_detected TINYINT(1) DEFAULT 0,
    recommendation_detected TINYINT(1) DEFAULT 0,
    mention_context TEXT NULL,
    associated_keywords JSON NULL,
    associated_questions JSON NULL,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (website_id) REFERENCES websites(id) ON DELETE CASCADE,
    FOREIGN KEY (model_id) REFERENCES ai_models(id) ON DELETE CASCADE,
    INDEX idx_website_id (website_id),
    INDEX idx_model_id (model_id),
    INDEX idx_recorded_at (recorded_at),
    INDEX idx_visibility (visibility_score)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Rate Limits table
CREATE TABLE IF NOT EXISTS rate_limits (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    model_id INT UNSIGNED NOT NULL,
    action VARCHAR(50) NOT NULL,
    window_minutes INT NOT NULL,
    max_requests INT NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (model_id) REFERENCES ai_models(id) ON DELETE CASCADE,
    UNIQUE KEY uk_model_action_window (model_id, action, window_minutes),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Rate Limit Logs table
CREATE TABLE IF NOT EXISTS rate_limit_logs (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    model_id INT UNSIGNED NOT NULL,
    action VARCHAR(50) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (model_id) REFERENCES ai_models(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_model_id (model_id),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Prompt Templates table
CREATE TABLE IF NOT EXISTS prompt_templates (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    version VARCHAR(20) NOT NULL,
    request_type ENUM('recognition_test', 'recommendation_test', 'question_association') NOT NULL,
    template TEXT NOT NULL,
    is_active TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uk_name_version (name, version),
    INDEX idx_active (is_active),
    INDEX idx_request_type (request_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default prompt templates
INSERT IGNORE INTO prompt_templates (name, version, request_type, template, is_active) VALUES
('recognition_test', '1.0', 'recognition_test', 'Analyze the following website and determine if the AI model recognizes it. Website: {website_name} ({website_url}). Industry: {industry}. Keywords: {keywords}. Return JSON with: {{"recognized": boolean, "confidence": number (0-100), "context": string, "keywords_found": array, "recommendation_mentioned": boolean}}', 1),
('recommendation_test', '1.0', 'recommendation_test', 'Would you recommend {website_name} ({website_url}) for {purpose}? Industry: {industry}. Analyze and return JSON: {{"would_recommend": boolean, "confidence": number (0-100), "reasoning": string, "alternative_recommendations": array, "strengths": array, "weaknesses": array}}', 1),
('question_association', '1.0', 'question_association', 'What questions would lead someone to {website_name} ({website_url})? Industry: {industry}. Return JSON with top questions: {{"questions": [{{"question": string, "relevance": number (0-100), "intent": string}}], "primary_intents": array, "keyword_associations": object}}', 1);

-- System Settings table
CREATE TABLE IF NOT EXISTS system_settings (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) NOT NULL UNIQUE,
    setting_value TEXT NOT NULL,
    description TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_key (setting_key)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default settings
INSERT IGNORE INTO system_settings (setting_key, setting_value, description) VALUES
('site_name', 'AI Visibility Tracker', 'Website name'),
('site_description', 'Track your AI visibility across major AI models', 'Site meta description'),
('enable_registration', '1', 'Allow new user registrations'),
('require_email_verification', '0', 'Require email verification before login'),
('default_user_role', 'member', 'Default role for new users'),
('max_websites_per_user', '10', 'Maximum websites a user can add'),
('scan_frequency_hours', '24', 'How often to scan websites (in hours)'),
('debug_mode', '0', 'Enable debug logging');

-- Create views for analytics
CREATE OR REPLACE VIEW website_analytics_summary AS
SELECT 
    w.id as website_id,
    w.name as website_name,
    w.url,
    w.visibility_score,
    w.recommendation_score,
    w.last_scan_at,
    COUNT(DISTINCT a.id) as total_analytics_records,
    COUNT(DISTINCT CASE WHEN a.mention_detected = 1 THEN a.id END) as mention_count,
    COUNT(DISTINCT CASE WHEN a.recommendation_detected = 1 THEN a.id END) as recommendation_count,
    AVG(a.visibility_score) as avg_visibility_score,
    MAX(a.recorded_at) as last_analytics_update
FROM websites w
LEFT JOIN analytics a ON w.id = a.website_id
WHERE w.is_active = 1
GROUP BY w.id;

CREATE OR REPLACE VIEW model_performance_summary AS
SELECT 
    m.id as model_id,
    m.name as model_name,
    m.display_name,
    COUNT(DISTINCT r.id) as total_requests,
    COUNT(DISTINCT CASE WHEN r.status = 'completed' THEN r.id END) as completed_requests,
    COUNT(DISTINCT CASE WHEN r.status = 'failed' THEN r.id END) as failed_requests,
    AVG(r.processing_time_ms) as avg_processing_time,
    AVG(CASE WHEN r.status = 'completed' THEN a.visibility_score END) as avg_visibility_score
FROM ai_models m
LEFT JOIN ai_requests r ON m.id = r.model_id
LEFT JOIN analytics a ON r.website_id = a.website_id AND r.model_id = a.model_id
WHERE m.is_active = 1
GROUP BY m.id;

-- Create indexes for performance
CREATE INDEX idx_analytics_composite ON analytics(website_id, model_id, recorded_at);
CREATE INDEX idx_ai_requests_composite ON ai_requests(website_id, model_id, created_at);
CREATE INDEX idx_job_queue_composite ON job_queue(website_id, status, created_at);