<?php

class Queue
{
    private static $instance = null;
    private $db;
    private $logger;
    private $rateLimiter;

    const STATUS_PENDING = 'pending';
    const STATUS_PROCESSING = 'processing';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';

    private function __construct()
    {
        $this->db = Database::getInstance();
        $this->logger = Logger::getInstance();
        $this->rateLimiter = new RateLimiter();
    }

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function addJob($type, $payload, $websiteId, $userId, $priority = 0)
    {
        $jobId = $this->db->execute(
            "INSERT INTO job_queue (type, payload, website_id, user_id, status, priority, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())",
            [$type, json_encode($payload), $websiteId, $userId, self::STATUS_PENDING, $priority]
        );

        $jobId = $this->db->lastInsertId();
        
        $this->logger->logInfo("Job added to queue", [
            'job_id' => $jobId,
            'type' => $type,
            'website_id' => $websiteId,
            'user_id' => $userId,
            'priority' => $priority
        ]);
        
        return $jobId;
    }

    public function processNextJob()
    {
        $this->db->beginTransaction();
        
        try {
            $job = $this->db->fetch(
                "SELECT id, type, payload, website_id, user_id, attempts FROM job_queue 
                 WHERE status = ? AND (scheduled_at IS NULL OR scheduled_at <= NOW())
                 ORDER BY priority DESC, created_at ASC 
                 LIMIT 1 FOR UPDATE",
                [self::STATUS_PENDING]
            );
            
            if (!$job) {
                $this->db->commit();
                return null;
            }
            
            $this->db->execute(
                "UPDATE job_queue SET status = ?, started_at = NOW(), attempts = attempts + 1 WHERE id = ?",
                [self::STATUS_PROCESSING, $job['id']]
            );
            
            $this->db->commit();
            
            $this->logger->logInfo("Job processing started", [
                'job_id' => $job['id'],
                'type' => $job['type'],
                'website_id' => $job['website_id']
            ]);
            
            return $job;
            
        } catch (Exception $e) {
            $this->db->rollBack();
            $this->logger->logError("Failed to process next job", [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function completeJob($jobId, $result = null)
    {
        $this->db->execute(
            "UPDATE job_queue SET status = ?, completed_at = NOW(), result = ? WHERE id = ?",
            [self::STATUS_COMPLETED, json_encode($result), $jobId]
        );
        
        $this->logger->logInfo("Job completed", [
            'job_id' => $jobId,
            'result' => $result
        ]);
    }

    public function failJob($jobId, $error = null)
    {
        $job = $this->db->fetch(
            "SELECT attempts, max_attempts FROM job_queue WHERE id = ?",
            [$jobId]
        );
        
        if ($job['attempts'] >= $job['max_attempts']) {
            $this->db->execute(
                "UPDATE job_queue SET status = ?, failed_at = NOW(), error_message = ? WHERE id = ?",
                [self::STATUS_FAILED, $error, $jobId]
            );
            
            $this->logger->logError("Job failed permanently", [
                'job_id' => $jobId,
                'error' => $error,
                'attempts' => $job['attempts']
            ]);
        } else {
            $delay = $this->calculateBackoffDelay($job['attempts']);
            
            $this->db->execute(
                "UPDATE job_queue SET status = ?, scheduled_at = DATE_ADD(NOW(), INTERVAL ? SECOND) WHERE id = ?",
                [self::STATUS_PENDING, $delay, $jobId]
            );
            
            $this->logger->logWarning("Job failed, retrying with backoff", [
                'job_id' => $jobId,
                'error' => $error,
                'attempts' => $job['attempts'],
                'delay' => $delay
            ]);
        }
    }

    private function calculateBackoffDelay($attempts)
    {
        $baseDelay = 60;
        $maxDelay = 3600;
        
        $delay = $baseDelay * pow(2, $attempts - 1);
        
        return min($delay, $maxDelay);
    }

    public function getJobStatus($jobId)
    {
        return $this->db->fetch(
            "SELECT status, result, error_message, created_at, started_at, completed_at FROM job_queue WHERE id = ?",
            [$jobId]
        );
    }

    public function getJobsByWebsite($websiteId, $limit = 50)
    {
        return $this->db->fetchAll(
            "SELECT id, type, status, priority, created_at, started_at, completed_at, result 
             FROM job_queue 
             WHERE website_id = ? 
             ORDER BY created_at DESC 
             LIMIT ?",
            [$websiteId, $limit]
        );
    }

    public function getJobsByUser($userId, $limit = 50)
    {
        return $this->db->fetchAll(
            "SELECT id, type, status, priority, created_at, started_at, completed_at, result 
             FROM job_queue 
             WHERE user_id = ? 
             ORDER BY created_at DESC 
             LIMIT ?",
            [$userId, $limit]
        );
    }

    public function clearCompletedJobs($olderThanDays = 7)
    {
        $this->db->execute(
            "DELETE FROM job_queue WHERE status IN (?, ?) AND completed_at < DATE_SUB(NOW(), INTERVAL ? DAY)",
            [self::STATUS_COMPLETED, self::STATUS_FAILED, $olderThanDays]
        );
        
        $this->logger->logInfo("Completed jobs cleaned up", ['older_than_days' => $olderThanDays]);
    }

    public function getQueueStats()
    {
        return $this->db->fetch(
            "SELECT 
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'processing' THEN 1 ELSE 0 END) as processing,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
             FROM job_queue"
        );
    }
}

class RateLimiter
{
    private $db;
    private $logger;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->logger = Logger::getInstance();
    }

    public function checkRateLimit($userId, $model, $action)
    {
        $limits = $this->getRateLimits($model, $action);
        
        foreach ($limits as $limit) {
            $window = $limit['window_minutes'] * 60;
            $maxRequests = $limit['max_requests'];
            
            $count = $this->db->fetch(
                "SELECT COUNT(*) as count 
                 FROM rate_limit_logs 
                 WHERE user_id = ? AND model = ? AND action = ? 
                 AND created_at > DATE_SUB(NOW(), INTERVAL ? SECOND)",
                [$userId, $model, $action, $window]
            );
            
            if ($count['count'] >= $maxRequests) {
                $this->logger->logWarning("Rate limit exceeded", [
                    'user_id' => $userId,
                    'model' => $model,
                    'action' => $action,
                    'limit' => $maxRequests,
                    'window' => $limit['window_minutes']
                ]);
                
                return false;
            }
        }
        
        $this->db->execute(
            "INSERT INTO rate_limit_logs (user_id, model, action, created_at) VALUES (?, ?, ?, NOW())",
            [$userId, $model, $action]
        ]);
        
        return true;
    }

    private function getRateLimits($model, $action)
    {
        $defaultLimits = [
            ['window_minutes' => 1, 'max_requests' => 5],
            ['window_minutes' => 60, 'max_requests' => 100],
            ['window_minutes' => 1440, 'max_requests' => 1000]
        ];
        
        $modelLimits = $this->db->fetchAll(
            "SELECT window_minutes, max_requests FROM rate_limits WHERE model = ? AND action = ? AND is_active = 1",
            [$model, $action]
        );
        
        return !empty($modelLimits) ? $modelLimits : $defaultLimits;
    }

    public function clearOldLogs($olderThanDays = 30)
    {
        $this->db->execute(
            "DELETE FROM rate_limit_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$olderThanDays]
        );
    }
}