<?php

class DashboardController extends BaseController
{
    private $websiteModel;
    private $aiService;

    public function __construct()
    {
        parent::__construct();
        $this->auth->requireLogin();
        $this->websiteModel = new Website();
        $this->aiService = new AIService();
    }

    public function index()
    {
        $userId = $this->auth->getUserId();
        
        $stats = $this->websiteModel->getStatsByUser($userId);
        
        $recentWebsites = $this->websiteModel->getByUserId($userId, 5);
        
        $recentJobs = $this->db->fetchAll(
            "SELECT j.*, w.name as website_name, m.display_name as model_name
             FROM job_queue j
             JOIN websites w ON j.website_id = w.id
             JOIN ai_models m ON JSON_UNQUOTE(JSON_EXTRACT(j.payload, '$.model_id')) = m.id
             WHERE j.user_id = ?
             ORDER BY j.created_at DESC
             LIMIT 10",
            [$userId]
        );
        
        echo $this->renderLayout('dashboard.index', [
            'page_title' => 'Dashboard - AI Visibility Tracker',
            'stats' => $stats,
            'recent_websites' => $recentWebsites,
            'recent_jobs' => $recentJobs
        ]);
    }

    public function websites()
    {
        $userId = $this->auth->getUserId();
        
        $page = (int)($_GET['page'] ?? 1);
        $perPage = 10;
        $offset = ($page - 1) * $perPage;
        
        $websites = $this->websiteModel->getByUserId($userId, $perPage, $offset);
        
        $totalCount = $this->db->fetch(
            "SELECT COUNT(*) as count FROM websites WHERE user_id = ?",
            [$userId]
        )['count'];
        
        $pagination = $this->paginate($totalCount, $page, $perPage, '/dashboard/websites');
        
        echo $this->renderLayout('dashboard.websites', [
            'page_title' => 'My Websites - AI Visibility Tracker',
            'websites' => $websites,
            'pagination' => $pagination
        ]);
    }

    public function addWebsite()
    {
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $name = $this->sanitizeInput($_POST['name'] ?? '');
            $url = $this->sanitizeInput($_POST['url'] ?? '');
            $description = $this->sanitizeInput($_POST['description'] ?? '');
            $industry = $this->sanitizeInput($_POST['industry'] ?? '');
            $targetKeywords = $this->sanitizeInput($_POST['target_keywords'] ?? '');
            
            $validationErrors = $this->validateInput($_POST, [
                'name' => 'required|max:255',
                'url' => 'required|url',
                'industry' => 'max:100'
            ]);
            
            if (empty($validationErrors)) {
                try {
                    $userId = $this->auth->getUserId();
                    
                    $websiteId = $this->websiteModel->create(
                        $userId,
                        $name,
                        $url,
                        $description,
                        $industry,
                        $targetKeywords
                    );
                    
                    $success = true;
                    $this->logger->logInfo("Website added", ['website_id' => $websiteId, 'user_id' => $userId]);
                    
                    $this->redirect("/dashboard/websites/{$websiteId}");
                    
                } catch (Exception $e) {
                    $errors[] = $e->getMessage();
                    $this->logger->logError("Failed to add website", ['error' => $e->getMessage(), 'user_id' => $this->auth->getUserId()]);
                }
            } else {
                $errors = array_merge($errors, array_flatten($validationErrors));
            }
        }
        
        $industries = $this->getIndustries();
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('dashboard.add-website', [
            'page_title' => 'Add Website - AI Visibility Tracker',
            'errors' => $errors,
            'success' => $success,
            'industries' => $industries,
            'csrf_token' => $csrfToken
        ]);
    }

    public function viewWebsite($websiteId)
    {
        $userId = $this->auth->getUserId();
        
        $website = $this->db->fetch(
            "SELECT w.*, 
                    (SELECT AVG(visibility_score) FROM analytics WHERE website_id = w.id) as avg_visibility_score,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND mention_detected = 1) as mention_count,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND recommendation_detected = 1) as recommendation_count
             FROM websites w 
             WHERE w.id = ? AND w.user_id = ?",
            [$websiteId, $userId]
        );
        
        if (!$website) {
            $this->redirect('/dashboard/websites');
        }
        
        $analytics = $this->db->fetchAll(
            "SELECT a.*, m.display_name as model_name
             FROM analytics a
             JOIN ai_models m ON a.model_id = m.id
             WHERE a.website_id = ?
             ORDER BY a.recorded_at DESC
             LIMIT 50",
            [$websiteId]
        );
        
        $models = $this->db->fetchAll(
            "SELECT id, display_name FROM ai_models WHERE is_active = 1"
        );
        
        echo $this->renderLayout('dashboard.view-website', [
            'page_title' => $website['name'] . ' - AI Visibility Tracker',
            'website' => $website,
            'analytics' => $analytics,
            'models' => $models
        ]);
    }

    public function runAITest($websiteId)
    {
        $this->validateCsrf();
        
        $userId = $this->auth->getUserId();
        
        $website = $this->db->fetch(
            "SELECT id FROM websites WHERE id = ? AND user_id = ?",
            [$websiteId, $userId]
        );
        
        if (!$website) {
            $this->jsonResponse(['error' => 'Website not found'], 404);
        }
        
        $testType = $_POST['test_type'] ?? '';
        $modelId = (int)($_POST['model_id'] ?? 0);
        
        if (!in_array($testType, ['recognition_test', 'recommendation_test', 'question_association'])) {
            $this->jsonResponse(['error' => 'Invalid test type'], 400);
        }
        
        if ($modelId <= 0) {
            $this->jsonResponse(['error' => 'Invalid model'], 400);
        }
        
        try {
            switch ($testType) {
                case 'recognition_test':
                    $jobId = $this->aiService->queueRecognitionTest($websiteId, $userId, $modelId);
                    break;
                case 'recommendation_test':
                    $purpose = $_POST['purpose'] ?? 'general use';
                    $jobId = $this->aiService->queueRecommendationTest($websiteId, $userId, $modelId, $purpose);
                    break;
                case 'question_association':
                    $jobId = $this->aiService->queueQuestionAssociation($websiteId, $userId, $modelId);
                    break;
            }
            
            $this->logger->logInfo("AI test queued", ['job_id' => $jobId, 'website_id' => $websiteId, 'user_id' => $userId]);
            
            $this->jsonResponse(['success' => true, 'job_id' => $jobId]);
            
        } catch (Exception $e) {
            $this->logger->logError("Failed to queue AI test", ['error' => $e->getMessage(), 'website_id' => $websiteId]);
            $this->jsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    public function deleteWebsite($websiteId)
    {
        $this->validateCsrf();
        
        $userId = $this->auth->getUserId();
        
        $website = $this->db->fetch(
            "SELECT id FROM websites WHERE id = ? AND user_id = ?",
            [$websiteId, $userId]
        );
        
        if (!$website) {
            $this->jsonResponse(['error' => 'Website not found'], 404);
        }
        
        try {
            $this->websiteModel->delete($websiteId);
            
            $this->logger->logInfo("Website deleted", ['website_id' => $websiteId, 'user_id' => $userId]);
            
            $this->jsonResponse(['success' => true]);
            
        } catch (Exception $e) {
            $this->logger->logError("Failed to delete website", ['error' => $e->getMessage(), 'website_id' => $websiteId]);
            $this->jsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    public function analytics()
    {
        $userId = $this->auth->getUserId();
        
        $summary = $this->db->fetch(
            "SELECT 
                COUNT(DISTINCT w.id) as total_websites,
                AVG(a.visibility_score) as avg_visibility_score,
                COUNT(CASE WHEN a.mention_detected = 1 THEN 1 END) as total_mentions,
                COUNT(CASE WHEN a.recommendation_detected = 1 THEN 1 END) as total_recommendations,
                COUNT(DISTINCT a.model_id) as models_analyzed
             FROM websites w
             LEFT JOIN analytics a ON w.id = a.website_id
             WHERE w.user_id = ?",
            [$userId]
        );
        
        $modelBreakdown = $this->db->fetchAll(
            "SELECT m.display_name, 
                    AVG(a.visibility_score) as avg_visibility,
                    COUNT(CASE WHEN a.mention_detected = 1 THEN 1 END) as mentions,
                    COUNT(CASE WHEN a.recommendation_detected = 1 THEN 1 END) as recommendations
             FROM analytics a
             JOIN ai_models m ON a.model_id = m.id
             JOIN websites w ON a.website_id = w.id
             WHERE w.user_id = ?
             GROUP BY m.id",
            [$userId]
        );
        
        $timeSeries = $this->db->fetchAll(
            "SELECT DATE(recorded_at) as date,
                    AVG(visibility_score) as avg_visibility,
                    COUNT(*) as total_records
             FROM analytics a
             JOIN websites w ON a.website_id = w.id
             WHERE w.user_id = ? AND recorded_at > DATE_SUB(NOW(), INTERVAL 30 DAY)
             GROUP BY DATE(recorded_at)
             ORDER BY date",
            [$userId]
        );
        
        echo $this->renderLayout('dashboard.analytics', [
            'page_title' => 'Analytics - AI Visibility Tracker',
            'summary' => $summary,
            'model_breakdown' => $modelBreakdown,
            'time_series' => $timeSeries
        ]);
    }

    public function settings()
    {
        $userId = $this->auth->getUserId();
        
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $currentPassword = $_POST['current_password'] ?? '';
            $newPassword = $_POST['new_password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';
            
            if (!empty($currentPassword) && !empty($newPassword)) {
                if ($newPassword !== $confirmPassword) {
                    $errors[] = 'New passwords do not match.';
                } else {
                    try {
                        $user = $this->db->fetch(
                            "SELECT password_hash FROM users WHERE id = ?",
                            [$userId]
                        );
                        
                        if (password_verify($currentPassword, $user['password_hash'])) {
                            $this->userModel = new User();
                            $this->userModel->updatePassword($userId, $newPassword);
                            $success = true;
                        } else {
                            $errors[] = 'Current password is incorrect.';
                        }
                    } catch (Exception $e) {
                        $errors[] = $e->getMessage();
                    }
                }
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('dashboard.settings', [
            'page_title' => 'Settings - AI Visibility Tracker',
            'errors' => $errors,
            'success' => $success,
            'csrf_token' => $csrfToken
        ]);
    }

    private function getIndustries()
    {
        return [
            'Technology' => 'Technology',
            'E-commerce' => 'E-commerce',
            'Healthcare' => 'Healthcare',
            'Finance' => 'Finance',
            'Education' => 'Education',
            'Real Estate' => 'Real Estate',
            'Travel & Tourism' => 'Travel & Tourism',
            'Food & Beverage' => 'Food & Beverage',
            'Entertainment' => 'Entertainment',
            'Professional Services' => 'Professional Services',
            'Manufacturing' => 'Manufacturing',
            'Retail' => 'Retail',
            'Marketing' => 'Marketing',
            'Other' => 'Other'
        ];
    }
}