<?php
/**
 * AI Visibility Tracker - Queue Worker
 * 
 * Run this script to process queued AI requests
 * Usage: php worker.php [worker-id]
 */

// Start session
session_start();

// Define base path
define('BASE_PATH', __DIR__);

// Check if application is installed
if (!file_exists(BASE_PATH . '/config.php')) {
    die("Application is not installed. Please run the installer first.\n");
}

// Load configuration
require_once BASE_PATH . '/config.php';

// Load core classes
require_once BASE_PATH . '/core/Logger.php';
require_once BASE_PATH . '/core/Database.php';
require_once BASE_PATH . '/core/Auth.php';
require_once BASE_PATH . '/core/Queue.php';

// Load models
require_once BASE_PATH . '/app/models/User.php';
require_once BASE_PATH . '/app/models/Website.php';

// Load services
require_once BASE_PATH . '/app/services/AIService.php';
require_once BASE_PATH . '/app/services/RateLimiterService.php';

// Initialize logger
$logger = Logger::getInstance();

// Get worker ID from command line
$workerId = $argv[1] ?? '1';

$logger->logInfo("Queue worker started", ['worker_id' => $workerId]);

// Set up signal handlers for graceful shutdown
pcntl_async_signals(true);

$shutdown = false;
pcntl_signal(SIGTERM, function() use (&$shutdown, $logger) {
    $shutdown = true;
    $logger->logInfo("Worker received SIGTERM, shutting down");
});

pcntl_signal(SIGINT, function() use (&$shutdown, $logger) {
    $shutdown = true;
    $logger->logInfo("Worker received SIGINT, shutting down");
});

// Main worker loop
$queue = Queue::getInstance();
$aiService = new AIService();

$jobsProcessed = 0;
$startTime = time();

try {
    while (!$shutdown) {
        try {
            // Process next job
            $job = $queue->processNextJob();
            
            if ($job) {
                $logger->logInfo("Processing job", [
                    'job_id' => $job['id'],
                    'type' => $job['type'],
                    'worker_id' => $workerId
                ]);
                
                // Process based on job type
                switch ($job['type']) {
                    case 'recognition_test':
                        $aiService->processRecognitionTest($job);
                        break;
                        
                    case 'recommendation_test':
                        $aiService->processRecommendationTest($job);
                        break;
                        
                    case 'question_association':
                        $aiService->processQuestionAssociation($job);
                        break;
                        
                    default:
                        $logger->logError("Unknown job type", [
                            'job_id' => $job['id'],
                            'type' => $job['type']
                        ]);
                        $queue->failJob($job['id'], 'Unknown job type: ' . $job['type']);
                }
                
                $jobsProcessed++;
                
                // Log progress every 10 jobs
                if ($jobsProcessed % 10 === 0) {
                    $uptime = time() - $startTime;
                    $logger->logInfo("Worker progress", [
                        'worker_id' => $workerId,
                        'jobs_processed' => $jobsProcessed,
                        'uptime_seconds' => $uptime,
                        'avg_jobs_per_minute' => ($jobsProcessed / $uptime) * 60
                    ]);
                }
                
            } else {
                // No jobs available, wait before checking again
                $logger->logDebug("No jobs available, waiting...", ['worker_id' => $workerId]);
                sleep(5);
            }
            
            // Clean up old completed jobs every hour
            if (time() % 3600 < 10) {
                $queue->clearCompletedJobs(7); // Keep jobs for 7 days
            }
            
        } catch (Exception $e) {
            $logger->logError("Worker error", [
                'worker_id' => $workerId,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            // Wait before retrying
            sleep(10);
        }
    }
    
} catch (Exception $e) {
    $logger->logError("Worker fatal error", [
        'worker_id' => $workerId,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
}

// Cleanup
$uptime = time() - $startTime;
$logger->logInfo("Worker shutting down", [
    'worker_id' => $workerId,
    'jobs_processed' => $jobsProcessed,
    'uptime_seconds' => $uptime,
    'avg_jobs_per_minute' => $uptime > 0 ? ($jobsProcessed / $uptime) * 60 : 0
]);

echo "Worker {$workerId} shutdown. Processed {$jobsProcessed} jobs in {$uptime} seconds.\n";