<?php

class AIService
{
    private $db;
    private $logger;
    private $queue;
    private $rateLimiter;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->logger = Logger::getInstance();
        $this->queue = Queue::getInstance();
        $this->rateLimiter = new RateLimiter();
    }

    public function queueRecognitionTest($websiteId, $userId, $modelId)
    {
        $this->logger->logInfo("Queueing recognition test", [
            'website_id' => $websiteId,
            'user_id' => $userId,
            'model_id' => $modelId
        ]);
        
        $website = $this->db->fetch(
            "SELECT name, url, industry, target_keywords FROM websites WHERE id = ?",
            [$websiteId]
        );
        
        if (!$website) {
            throw new Exception("Website not found.");
        }
        
        $payload = [
            'model_id' => $modelId,
            'website_name' => $website['name'],
            'website_url' => $website['url'],
            'industry' => $website['industry'],
            'keywords' => $website['target_keywords']
        ];
        
        $jobId = $this->queue->addJob('recognition_test', $payload, $websiteId, $userId);
        
        $this->logger->logInfo("Recognition test queued", ['job_id' => $jobId]);
        
        return $jobId;
    }

    public function queueRecommendationTest($websiteId, $userId, $modelId, $purpose)
    {
        $this->logger->logInfo("Queueing recommendation test", [
            'website_id' => $websiteId,
            'user_id' => $userId,
            'model_id' => $modelId,
            'purpose' => $purpose
        ]);
        
        $website = $this->db->fetch(
            "SELECT name, url, industry FROM websites WHERE id = ?",
            [$websiteId]
        );
        
        if (!$website) {
            throw new Exception("Website not found.");
        }
        
        $payload = [
            'model_id' => $modelId,
            'website_name' => $website['name'],
            'website_url' => $website['url'],
            'industry' => $website['industry'],
            'purpose' => $purpose
        ];
        
        $jobId = $this->queue->addJob('recommendation_test', $payload, $websiteId, $userId);
        
        $this->logger->logInfo("Recommendation test queued", ['job_id' => $jobId]);
        
        return $jobId;
    }

    public function queueQuestionAssociation($websiteId, $userId, $modelId)
    {
        $this->logger->logInfo("Queueing question association test", [
            'website_id' => $websiteId,
            'user_id' => $userId,
            'model_id' => $modelId
        ]);
        
        $website = $this->db->fetch(
            "SELECT name, url, industry FROM websites WHERE id = ?",
            [$websiteId]
        );
        
        if (!$website) {
            throw new Exception("Website not found.");
        }
        
        $payload = [
            'model_id' => $modelId,
            'website_name' => $website['name'],
            'website_url' => $website['url'],
            'industry' => $website['industry']
        ];
        
        $jobId = $this->queue->addJob('question_association', $payload, $websiteId, $userId);
        
        $this->logger->logInfo("Question association test queued", ['job_id' => $jobId]);
        
        return $jobId;
    }

    public function processRecognitionTest($job)
    {
        $this->logger->logInfo("Processing recognition test", ['job_id' => $job['id']]);
        
        $payload = json_decode($job['payload'], true);
        $startTime = microtime(true);
        
        try {
            if (!$this->rateLimiter->checkRateLimit($job['user_id'], $payload['model_id'], 'recognition_test')) {
                throw new Exception("Rate limit exceeded.");
            }
            
            $prompt = $this->buildRecognitionPrompt($payload);
            $response = $this->callAIAPI($payload['model_id'], $prompt);
            
            $processingTime = round((microtime(true) - $startTime) * 1000);
            
            $this->saveAIRequest(
                $job['website_id'],
                $job['user_id'],
                $payload['model_id'],
                'recognition_test',
                $prompt,
                $response,
                $processingTime
            );
            
            $result = $this->parseAIResponse($response);
            
            $this->saveAnalytics($job['website_id'], $payload['model_id'], $result);
            
            $this->updateWebsiteScores($job['website_id']);
            
            $this->queue->completeJob($job['id'], $result);
            
            $this->logger->logInfo("Recognition test completed", ['job_id' => $job['id']]);
            
        } catch (Exception $e) {
            $this->logger->logError("Recognition test failed", [
                'job_id' => $job['id'],
                'error' => $e->getMessage()
            ]);
            
            $this->queue->failJob($job['id'], $e->getMessage());
        }
    }

    public function processRecommendationTest($job)
    {
        $this->logger->logInfo("Processing recommendation test", ['job_id' => $job['id']]);
        
        $payload = json_decode($job['payload'], true);
        $startTime = microtime(true);
        
        try {
            if (!$this->rateLimiter->checkRateLimit($job['user_id'], $payload['model_id'], 'recommendation_test')) {
                throw new Exception("Rate limit exceeded.");
            }
            
            $prompt = $this->buildRecommendationPrompt($payload);
            $response = $this->callAIAPI($payload['model_id'], $prompt);
            
            $processingTime = round((microtime(true) - $startTime) * 1000);
            
            $this->saveAIRequest(
                $job['website_id'],
                $job['user_id'],
                $payload['model_id'],
                'recommendation_test',
                $prompt,
                $response,
                $processingTime
            );
            
            $result = $this->parseAIResponse($response);
            
            $analyticsData = [
                'recommendation_detected' => $result['would_recommend'] ?? false,
                'confidence' => $result['confidence'] ?? 0,
                'reasoning' => $result['reasoning'] ?? ''
            ];
            
            $this->saveAnalytics($job['website_id'], $payload['model_id'], $analyticsData);
            
            $this->updateWebsiteScores($job['website_id']);
            
            $this->queue->completeJob($job['id'], $result);
            
            $this->logger->logInfo("Recommendation test completed", ['job_id' => $job['id']]);
            
        } catch (Exception $e) {
            $this->logger->logError("Recommendation test failed", [
                'job_id' => $job['id'],
                'error' => $e->getMessage()
            ]);
            
            $this->queue->failJob($job['id'], $e->getMessage());
        }
    }

    public function processQuestionAssociation($job)
    {
        $this->logger->logInfo("Processing question association test", ['job_id' => $job['id']]);
        
        $payload = json_decode($job['payload'], true);
        $startTime = microtime(true);
        
        try {
            if (!$this->rateLimiter->checkRateLimit($job['user_id'], $payload['model_id'], 'question_association')) {
                throw new Exception("Rate limit exceeded.");
            }
            
            $prompt = $this->buildQuestionPrompt($payload);
            $response = $this->callAIAPI($payload['model_id'], $prompt);
            
            $processingTime = round((microtime(true) - $startTime) * 1000);
            
            $this->saveAIRequest(
                $job['website_id'],
                $job['user_id'],
                $payload['model_id'],
                'question_association',
                $prompt,
                $response,
                $processingTime
            );
            
            $result = $this->parseAIResponse($response);
            
            $analyticsData = [
                'associated_questions' => $result['questions'] ?? [],
                'primary_intents' => $result['primary_intents'] ?? [],
                'keyword_associations' => $result['keyword_associations'] ?? []
            ];
            
            $this->saveAnalytics($job['website_id'], $payload['model_id'], $analyticsData);
            
            $this->queue->completeJob($job['id'], $result);
            
            $this->logger->logInfo("Question association test completed", ['job_id' => $job['id']]);
            
        } catch (Exception $e) {
            $this->logger->logError("Question association test failed", [
                'job_id' => $job['id'],
                'error' => $e->getMessage()
            ]);
            
            $this->queue->failJob($job['id'], $e->getMessage());
        }
    }

    private function buildRecognitionPrompt($payload)
    {
        $template = $this->getPromptTemplate('recognition_test');
        
        $replacements = [
            '{website_name}' => $payload['website_name'],
            '{website_url}' => $payload['website_url'],
            '{industry}' => $payload['industry'] ?? 'general',
            '{keywords}' => $payload['keywords'] ?? 'N/A'
        ];
        
        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }

    private function buildRecommendationPrompt($payload)
    {
        $template = $this->getPromptTemplate('recommendation_test');
        
        $replacements = [
            '{website_name}' => $payload['website_name'],
            '{website_url}' => $payload['website_url'],
            '{industry}' => $payload['industry'] ?? 'general',
            '{purpose}' => $payload['purpose'] ?? 'your needs'
        ];
        
        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }

    private function buildQuestionPrompt($payload)
    {
        $template = $this->getPromptTemplate('question_association');
        
        $replacements = [
            '{website_name}' => $payload['website_name'],
            '{website_url}' => $payload['website_url'],
            '{industry}' => $payload['industry'] ?? 'general'
        ];
        
        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }

    private function getPromptTemplate($type)
    {
        $template = $this->db->fetch(
            "SELECT template FROM prompt_templates WHERE request_type = ? AND is_active = 1 ORDER BY version DESC LIMIT 1",
            [$type]
        );
        
        if (!$template) {
            return $this->getDefaultPromptTemplate($type);
        }
        
        return $template['template'];
    }

    private function getDefaultPromptTemplate($type)
    {
        $templates = [
            'recognition_test' => 'Analyze the following website and determine if you recognize it. Website: {website_name} ({website_url}). Industry: {industry}. Keywords: {keywords}. Return JSON with: {"recognized": boolean, "confidence": number (0-100), "context": string, "keywords_found": array, "recommendation_mentioned": boolean}',
            'recommendation_test' => 'Would you recommend {website_name} ({website_url}) for {purpose}? Industry: {industry}. Analyze and return JSON: {"would_recommend": boolean, "confidence": number (0-100), "reasoning": string, "alternative_recommendations": array, "strengths": array, "weaknesses": array}',
            'question_association' => 'What questions would lead someone to {website_name} ({website_url})? Industry: {industry}. Return JSON with top questions: {"questions": [{"question": string, "relevance": number (0-100), "intent": string}], "primary_intents": array, "keyword_associations": object}'
        ];
        
        return $templates[$type] ?? '';
    }

    private function callAIAPI($modelId, $prompt)
    {
        $model = $this->db->fetch(
            "SELECT name, api_endpoint FROM ai_models WHERE id = ? AND is_active = 1",
            [$modelId]
        );
        
        if (!$model) {
            throw new Exception("AI model not found or inactive.");
        }
        
        $apiKey = defined('OPENROUTER_API_KEY') ? OPENROUTER_API_KEY : '';
        
        if (empty($apiKey)) {
            throw new Exception("OpenRouter API key not configured.");
        }
        
        $ch = curl_init();
        
        $payload = [
            'model' => $model['name'],
            'messages' => [
                ['role' => 'user', 'content' => $prompt]
            ],
            'temperature' => 0.7,
            'max_tokens' => 1000
        ];
        
        curl_setopt_array($ch, [
            CURLOPT_URL => 'https://openrouter.ai/api/v1/chat/completions',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $apiKey,
                'Content-Type: application/json',
                'HTTP-Referer: ' . BASE_URL,
                'X-Title: AI Visibility Tracker'
            ],
            CURLOPT_TIMEOUT => 60,
            CURLOPT_CONNECTTIMEOUT => 10
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception("API request failed: " . $error);
        }
        
        if ($httpCode !== 200) {
            throw new Exception("API returned HTTP code: " . $httpCode);
        }
        
        $data = json_decode($response, true);
        
        if (!isset($data['choices'][0]['message']['content'])) {
            throw new Exception("Invalid API response format.");
        }
        
        return $data['choices'][0]['message']['content'];
    }

    private function parseAIResponse($response)
    {
        $jsonMatch = [];
        preg_match('/\{[^{}]*\}/', $response, $jsonMatch);
        
        if (empty($jsonMatch)) {
            return ['error' => 'No JSON found in response', 'raw_response' => $response];
        }
        
        $json = json_decode($jsonMatch[0], true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return ['error' => 'Invalid JSON: ' . json_last_error_msg(), 'raw_response' => $response];
        }
        
        return $json;
    }

    private function saveAIRequest($websiteId, $userId, $modelId, $requestType, $prompt, $response, $processingTime)
    {
        $this->db->execute(
            "INSERT INTO ai_requests 
             (website_id, user_id, model_id, request_type, prompt_version, prompt_text, response_text, response_hash, processing_time_ms, status, created_at) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())",
            [
                $websiteId,
                $userId,
                $modelId,
                $requestType,
                '1.0',
                $prompt,
                $response,
                hash('sha256', $response),
                $processingTime,
                'completed'
            ]
        );
    }

    private function saveAnalytics($websiteId, $modelId, $data)
    {
        $visibilityScore = $data['confidence'] ?? 0;
        $mentionDetected = isset($data['recognized']) ? (bool)$data['recognized'] : false;
        $recommendationDetected = isset($data['would_recommend']) ? (bool)$data['would_recommend'] : false;
        $mentionContext = $data['context'] ?? $data['reasoning'] ?? '';
        $associatedKeywords = isset($data['keywords_found']) ? json_encode($data['keywords_found']) : null;
        $associatedQuestions = isset($data['questions']) ? json_encode($data['questions']) : null;
        
        $this->db->execute(
            "INSERT INTO analytics 
             (website_id, model_id, visibility_score, mention_detected, recommendation_detected, mention_context, associated_keywords, associated_questions, recorded_at) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())",
            [
                $websiteId,
                $modelId,
                $visibilityScore,
                $mentionDetected ? 1 : 0,
                $recommendationDetected ? 1 : 0,
                $mentionContext,
                $associatedKeywords,
                $associatedQuestions
            ]
        );
    }

    private function updateWebsiteScores($websiteId)
    {
        $scores = $this->db->fetch(
            "SELECT 
                AVG(visibility_score) as avg_visibility,
                AVG(CASE WHEN recommendation_detected = 1 THEN visibility_score ELSE NULL END) as avg_recommendation
             FROM analytics 
             WHERE website_id = ?",
            [$websiteId]
        );
        
        $visibilityScore = round($scores['avg_visibility'] ?? 0);
        $recommendationScore = round($scores['avg_recommendation'] ?? 0);
        
        $this->db->execute(
            "UPDATE websites SET 
             visibility_score = ?, 
             recommendation_score = ?, 
             updated_at = NOW() 
             WHERE id = ?",
            [$visibilityScore, $recommendationScore, $websiteId]
        );
    }
}